<?php
// src/MT5WebAPI.php
class MT5WebAPI {
    private $server;
    private $login;
    private $password;
    private $token = null;
    private $curl = null;

    public function __construct() {
        $this->server = $_ENV['MT5_SERVER'] ?? getenv('MT5_SERVER') ?: 'your-mt5-server.com:443';
        $this->login = $_ENV['MT5_MANAGER_LOGIN'] ?? getenv('MT5_MANAGER_LOGIN') ?: '';
        $this->password = $_ENV['MT5_MANAGER_PASSWORD'] ?? getenv('MT5_MANAGER_PASSWORD') ?: '';
        
        // Initialize persistent cURL handle
        $this->curl = curl_init();
        curl_setopt_array($this->curl, [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_MAXCONNECTS => 1,
            CURLOPT_HTTPHEADER => ['Connection: Keep-Alive'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => true, // Include headers in response
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT => 'MT5-WebAPI-Client/1.0',
            CURLOPT_COOKIEFILE => '', // Enable cookie handling
            CURLOPT_COOKIEJAR => '', // Enable cookie handling
        ]);
    }
    
    public function __destruct() {
        if ($this->curl) {
            curl_close($this->curl);
        }
    }

    private function request($endpoint, $params = [], $method = 'GET', $requireAuth = true) {
        if (!$this->token && $requireAuth) {
            $this->authenticate();
        }
        
        $url = "https://{$this->server}{$endpoint}";
        
        // Add query params for GET
        if ($method === 'GET' && !empty($params)) {
            $url .= '?' . http_build_query($params);
        }
        
        curl_setopt($this->curl, CURLOPT_URL, $url);
        curl_setopt($this->curl, CURLOPT_POST, $method === 'POST');
        
        if ($method === 'POST' && !empty($params)) {
            curl_setopt($this->curl, CURLOPT_POSTFIELDS, http_build_query($params));
        }
        
        $response = curl_exec($this->curl);
        $httpCode = curl_getinfo($this->curl, CURLINFO_HTTP_CODE);
        $curlError = curl_error($this->curl);
        
        if ($curlError) {
            return [null, "cURL Error: $curlError"];
        }
        
        // Split headers and body (since CURLOPT_HEADER is true)
        $headerSize = curl_getinfo($this->curl, CURLINFO_HEADER_SIZE);
        $body = substr($response, $headerSize);
        
        if ($httpCode !== 200) {
            return [null, "HTTP Error: $httpCode - Response: $body"];
        }
        
        $data = json_decode($body, true);
        
        if (!$data) {
            return [null, "Invalid JSON response: $body"];
        }
        
        // Check retcode
        if (isset($data['retcode']) && !preg_match('/^0\s/', $data['retcode']) && (int)$data['retcode'] != 0) {
            return [null, "MT5 API Error: {$data['retcode']}"];
        }
        
        return [$data, null];
    }

    private function authenticate() {
        // Step 1: Auth start
        $build = '4330'; // MT5 build version - must match working file
        $agent = 'WebAPI'; // Agent name - must match working file
        
        $path = '/api/auth/start?version=' . $build . '&agent=' . urlencode($agent) . 
                '&login=' . $this->login . '&type=manager';
        
        list($data, $error) = $this->request($path, [], 'GET', false); // false = don't require auth
        
        if ($error) {
            throw new Exception("Auth start failed: " . $error);
        }
        
        if (!isset($data['retcode']) || (int)$data['retcode'] != 0) {
            throw new Exception("Auth start error: " . ($data['retcode'] ?? 'Unknown'));
        }
        
        if (!isset($data['srv_rand'])) {
            throw new Exception("No srv_rand in auth response");
        }
        
        // Step 2: Process challenge
        // Convert password to UTF-16LE and create first MD5 hash
        $password_utf16 = mb_convert_encoding($this->password, 'UTF-16LE', 'UTF-8');
        $password_md5 = md5($password_utf16, true);
        
        // Append "WebAPI" as bytes
        $webapi_bytes = 'WebAPI';
        $password_hash_input = $password_md5 . $webapi_bytes;
        
        // Create final password hash
        $password_hash = md5($password_hash_input, true);
        
        // Convert hex srv_rand to binary
        $srv_rand_bin = hex2bin($data['srv_rand']);
        if ($srv_rand_bin === false) {
            throw new Exception("Failed to convert srv_rand from hex");
        }
        
        // Calculate srv_rand_answer = MD5(password_hash + srv_rand_bin)
        $srv_rand_answer_input = $password_hash . $srv_rand_bin;
        $srv_rand_answer = md5($srv_rand_answer_input);
        
        // Generate client random
        $cli_rand_buf = random_bytes(16);
        $cli_rand = bin2hex($cli_rand_buf);
        
        // Step 3: Auth answer
        $path = '/api/auth/answer?srv_rand_answer=' . $srv_rand_answer . '&cli_rand=' . $cli_rand;
        list($data, $error) = $this->request($path, [], 'GET', false); // false = don't require auth
        
        if ($error) {
            throw new Exception("Auth answer failed: " . $error);
        }
        
        if (!isset($data['retcode']) || (int)$data['retcode'] != 0) {
            throw new Exception("Auth answer error: " . ($data['retcode'] ?? 'Unknown'));
        }
        
        if (!isset($data['cli_rand_answer'])) {
            throw new Exception("No cli_rand_answer in auth response");
        }
        
        // Step 4: Verify server response
        $cli_rand_answer_input = $password_hash . $cli_rand_buf;
        $expected_cli_rand_answer = md5($cli_rand_answer_input);
        
        if ($expected_cli_rand_answer != $data['cli_rand_answer']) {
            throw new Exception("Auth answer error: invalid client answer");
        }
        
        $this->token = 'authenticated'; // Mark as authenticated
    }

    public function getLogins($group = '*') {
        list($data, $error) = $this->request('/api/user/logins', [
            'group' => $group
        ], 'GET');
        
        if ($error) {
            throw new Exception($error);
        }
        
        return $data['answer'] ?? [];
    }

    public function getAccounts($group = '*') {
        // Get list of logins from your broker's API
        $logins = $this->getLogins($group);
        
        if (empty($logins)) {
            return [];
        }
        
        // Get first 10 accounts for testing (to avoid too many requests)
        $logins = array_slice($logins, 0, 10);
        
        // Get user details for each login
        $accounts = [];
        foreach ($logins as $login) {
            list($data, $error) = $this->request('/api/user/get', [
                'login' => $login
            ], 'GET');
            
            if (!$error && isset($data['answer'])) {
                $userInfo = is_array($data['answer']) && isset($data['answer'][0]) ? $data['answer'][0] : $data['answer'];
                $accounts[] = $userInfo;
            }
        }
        
        return $accounts;
    }

    public function getAccountInfo($login) {
        list($data, $error) = $this->request('/api/user/get', [
            'login' => $login
        ], 'GET');
        
        if ($error) {
            return null;
        }
        
        $answer = $data['answer'] ?? null;
        return is_array($answer) && isset($answer[0]) ? $answer[0] : $answer;
    }

    public function getPositions($login) {
        list($data, $error) = $this->request('/api/position/get_page', [
            'login' => $login,
            'offset' => 0,
            'total' => 100
        ]);
        
        if ($error) {
            return [];
        }
        
        return $data['answer'] ?? [];
    }

    public function getDeals($login, $from = null, $to = null) {
        $params = ['login' => $login];
        if ($from) $params['from'] = $from;
        if ($to) $params['to'] = $to;
        
        list($data, $error) = $this->request('/api/deal/get_page', $params);
        
        if ($error) {
            return [];
        }
        
        return $data['answer'] ?? [];
    }

    /**
     * Get clients filtered by group
     * @param string $group Group filter (e.g., 'skylinkscapital\A\Basic')
     * @return array Array of client records
     */
    public function getClients($group = null) {
        if ($group === null) {
            $group = $_ENV['MT5_CLIENT_GROUP'] ?? getenv('MT5_CLIENT_GROUP') ?? 'skylinkscapital\\A\\Basic';
        }
        list($data, $error) = $this->request('/api/client/get', [
            'group' => $group
        ], 'GET');
        
        if ($error) {
            return [];
        }
        
        return $data['answer'] ?? [];
    }

    /**
     * Get account numbers (logins) for specific clients
     * @param array $clientIds Array of client IDs
     * @return array Associative array [clientId => [login1, login2, ...]]
     */
    public function getClientLogins($clientIds) {
        if (empty($clientIds)) {
            return [];
        }
        
        $clientIdString = is_array($clientIds) ? implode(',', $clientIds) : $clientIds;
        
        list($data, $error) = $this->request('/api/client/user/get_logins', [
            'client' => $clientIdString
        ], 'GET');
        
        if ($error) {
            return [];
        }
        
        return $data['answer'] ?? [];
    }

    /**
     * Get clients with their associated account numbers
     * @param string $group Group filter
     * @return array Array of clients with login numbers
     */
    public function getClientsWithAccounts($group = 'skylinkscapital\\A\\Basic') {
        $clients = $this->getClients($group);
        
        if (empty($clients)) {
            return [];
        }
        
        // Extract client IDs
        $clientIds = array_map(function($client) {
            return $client['RecordID'] ?? null;
        }, $clients);
        $clientIds = array_filter($clientIds);
        
        // Get account numbers for all clients
        $clientLogins = $this->getClientLogins($clientIds);
        
        // Merge data
        $result = [];
        foreach ($clients as $client) {
            $clientId = $client['RecordID'] ?? null;
            if ($clientId) {
                $client['Logins'] = $clientLogins[$clientId] ?? [];
                $result[] = $client;
            }
        }
        
        return $result;
    }
}
